/**
 * Chromecast Helper for Helge Haukeland CDN
 * Ensures authentication tokens are passed through to Chromecast devices
 */
class ChromecastHelper {
    constructor(options = {}) {
        this.options = Object.assign({
            authTokenName: 'auth_token',
            debug: false
        }, options);
        
        this.initialize();
    }
    
    initialize() {
        if (this.options.debug) {
            console.log('Initializing ChromecastHelper');
        }
        
        // Ensure Cast API is loaded
        window.__onGCastApiAvailable = (isAvailable) => {
            if (isAvailable) {
                this.initializeCastApi();
            }
        };
        
        // Load Google Cast API if not already loaded
        if (!document.querySelector('script[src*="cast_sender.js"]')) {
            const script = document.createElement('script');
            script.src = 'https://www.gstatic.com/cv/js/sender/v1/cast_sender.js?loadCastFramework=1';
            document.head.appendChild(script);
        }
    }
    
    initializeCastApi() {
        const context = cast.framework.CastContext.getInstance();
        context.setOptions({
            receiverApplicationId: chrome.cast.media.DEFAULT_MEDIA_RECEIVER_APP_ID,
            autoJoinPolicy: chrome.cast.AutoJoinPolicy.ORIGIN_SCOPED
        });
        
        // Hook into session start event
        context.addEventListener(
            cast.framework.CastContextEventType.SESSION_STATE_CHANGED,
            this.sessionStateChangeHandler.bind(this)
        );
        
        if (this.options.debug) {
            console.log('Cast API initialized');
        }
    }
    
    sessionStateChangeHandler(event) {
        if (this.options.debug) {
            console.log('Cast session state changed:', event);
        }
        
        // When a session is about to start, we patch any video elements
        // to include the auth token in their src attributes
        if (event.sessionState === cast.framework.SessionState.SESSION_STARTING) {
            this.prepareVideosForCasting();
        }
    }
    
    prepareVideosForCasting() {
        // Get auth token from localStorage or cookie
        const authToken = this.getAuthToken();
        
        if (!authToken) {
            console.error('No auth token found for casting');
            return;
        }
        
        // Find all video elements on the page
        const videos = document.querySelectorAll('video');
        
        videos.forEach(video => {
            // Store the original src if we haven't already
            if (!video.dataset.originalSrc) {
                video.dataset.originalSrc = video.src;
            }
            
            // Append auth token to the source URL
            const url = new URL(video.src);
            url.searchParams.set(this.options.authTokenName, authToken);
            
            // Update the video src
            video.src = url.toString();
            
            if (this.options.debug) {
                console.log('Updated video src for casting:', video.src);
            }
        });
    }
    
    getAuthToken() {
        // Try to get token from various sources
        // 1. Local Storage
        let authToken = localStorage.getItem(this.options.authTokenName);
        
        // 2. Check cookies if not found in localStorage
        if (!authToken) {
            authToken = this.getCookie(this.options.authTokenName);
        }
        
        // 3. Try to extract it from any API calls if possible
        if (!authToken && window.fetch) {
            // This would be more complex and depend on your app's architecture
            // Check if you're storing the token in another named variable
            authToken = localStorage.getItem('token') || 
                        localStorage.getItem('jwt') || 
                        localStorage.getItem('access_token');
        }
        
        return authToken;
    }
    
    getCookie(name) {
        const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
        return match ? match[2] : null;
    }
    
    /**
     * Utility method to update the URL of a specific video element
     * @param {HTMLVideoElement} videoElement - The video element to update
     * @param {string} customAuthToken - Optional custom auth token
     */
    updateVideoUrl(videoElement, customAuthToken = null) {
        if (!videoElement) return;
        
        const authToken = customAuthToken || this.getAuthToken();
        
        if (!authToken) {
            console.error('No auth token available for updating video URL');
            return;
        }
        
        const url = new URL(videoElement.src);
        url.searchParams.set(this.options.authTokenName, authToken);
        videoElement.src = url.toString();
        
        if (this.options.debug) {
            console.log('Manually updated video URL with auth token:', videoElement.src);
        }
    }
    
    /**
     * Create a properly authenticated media info object for casting
     * @param {string} contentUrl - URL of the media to cast
     * @param {string} contentType - MIME type of the media
     * @param {Object} metadata - Cast metadata
     * @returns {chrome.cast.media.MediaInfo}
     */
    createMediaInfo(contentUrl, contentType, metadata = null) {
        const authToken = this.getAuthToken();
        
        // Ensure the URL includes the auth token
        const url = new URL(contentUrl);
        url.searchParams.set(this.options.authTokenName, authToken);
        const authenticatedUrl = url.toString();
        
        // Create media info
        const mediaInfo = new chrome.cast.media.MediaInfo(authenticatedUrl, contentType);
        
        // Set metadata if provided
        if (metadata) {
            if (contentType.startsWith('video')) {
                mediaInfo.metadata = new chrome.cast.media.MovieMediaMetadata();
            } else if (contentType.startsWith('audio')) {
                mediaInfo.metadata = new chrome.cast.media.MusicTrackMediaMetadata();
            }
            
            // Apply metadata properties
            Object.assign(mediaInfo.metadata, metadata);
        }
        
        // Add custom headers if needed for your server
        mediaInfo.customData = {
            headers: {
                Authorization: `Bearer ${authToken}`
            }
        };
        
        return mediaInfo;
    }
}

// Auto-initialize when included
document.addEventListener('DOMContentLoaded', () => {
    window.chromecastHelper = new ChromecastHelper({ debug: true });
});

// Export for module usage
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ChromecastHelper;
} 