<?php
// Error logging setup
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', '/home/natasun/helgehaukeland.com/secure_storage/logs/php-error.log');

// Set upload limits at runtime
ini_set('upload_max_filesize', '4000M');
ini_set('post_max_size', '4000M');
ini_set('memory_limit', '1024M');
ini_set('max_execution_time', '1800');
ini_set('max_input_time', '1800');

// Debug logging
error_log("=== New Request ===");
error_log("Method: " . $_SERVER['REQUEST_METHOD']);
error_log("Origin: " . ($_SERVER['HTTP_ORIGIN'] ?? 'none'));
error_log("Auth: " . ($_SERVER['HTTP_AUTHORIZATION'] ?? 'none'));
error_log("Request URI: " . $_SERVER['REQUEST_URI']);
error_log("Upload max filesize: " . ini_get('upload_max_filesize'));
error_log("Post max size: " . ini_get('post_max_size'));
error_log("Memory limit: " . ini_get('memory_limit'));
error_log("Max execution time: " . ini_get('max_execution_time'));

// Remove the global content type header
// header('Content-Type: application/json');

require_once 'auth.php';
$auth = new Auth();

// At the top of the file, after error reporting
if (strpos($_SERVER['REQUEST_URI'], '/secure_storage/') !== false && 
    !strpos($_SERVER['SCRIPT_NAME'], 'index.php')) {
    header('HTTP/1.0 403 Forbidden');
    exit('Access Denied');
}

function formatSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $size = $bytes;
    $unitIndex = 0;
    
    while ($size >= 1024 && $unitIndex < count($units) - 1) {
        $size /= 1024;
        $unitIndex++;
    }
    
    return round($size, 1) . ' ' . $units[$unitIndex];
}

function validateFile($file, $type) {
    $config = require 'config.php';
    
    // Check file size
    if ($file['size'] > $config['max_file_size']) {
        return [
            'valid' => false,
            'error' => 'File size exceeds limit of ' . formatSize($config['max_file_size'])
        ];
    }
    
    // Check file type
    $mimeType = mime_content_type($file['tmp_name']);
    if (!in_array($mimeType, $config['allowed_types'])) {
        return [
            'valid' => false,
            'error' => 'Invalid file type. Allowed types: ' . implode(', ', $config['allowed_types'])
        ];
    }
    
    // Check storage limit
    $stats = getStorageStats();
    if ($stats['totalSize'] + $file['size'] > $config['storage_limit']) {
        return [
            'valid' => false,
            'error' => 'Storage limit exceeded'
        ];
    }
    
    return ['valid' => true];
}

function handleUpload() {
    header('Content-Type: application/json');
    error_log("Starting handleUpload");
    
    if (!isset($_FILES['file'])) {
        error_log("No file uploaded");
        http_response_code(400);
        echo json_encode(['error' => 'No file uploaded']);
        return;
    }

    $type = $_POST['type'] ?? 'video';
    $file = $_FILES['file'];
    
    // Validate file
    $validation = validateFile($file, $type);
    if (!$validation['valid']) {
        http_response_code(400);
        echo json_encode(['error' => $validation['error']]);
        return;
    }
    
    // Continue with upload...
    $uploadDir = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos') . '/';
    
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $newFileName = uniqid() . '_' . basename($file['name']);
    $targetPath = $uploadDir . $newFileName;
    
    if (move_uploaded_file($file['tmp_name'], $targetPath)) {
        error_log("File uploaded successfully");
        echo json_encode(['fileName' => $newFileName]);
    } else {
        error_log("Upload failed: " . print_r(error_get_last(), true));
        http_response_code(500);
        echo json_encode(['error' => 'Failed to upload file']);
    }
}

function handleFileRequest() {
    error_log("Starting handleFileRequest");
    
    $file = $_GET['file'] ?? '';
    $type = $_GET['type'] ?? 'video';
    $isDownload = isset($_GET['download']) && $_GET['download'] === 'true';
    $authToken = $_GET['auth_token'] ?? '';
    
    error_log("Requested file: " . $file);
    error_log("File type: " . $type);
    error_log("Is download: " . ($isDownload ? 'yes' : 'no'));
    error_log("Auth token in query: " . ($authToken ? 'yes' : 'no'));
    
    if (empty($file)) {
        header('Content-Type: application/json');
        http_response_code(400);
        echo json_encode(['error' => 'Missing file parameter']);
        exit;
    }

    // Clean filename and prevent directory traversal
    $file = basename($file);
    $filePath = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos') . '/' . $file;
    
    error_log("Looking for file at: " . $filePath);
    
    if (!file_exists($filePath)) {
        header('Content-Type: application/json');
        error_log("File not found: " . $filePath);
        http_response_code(404);
        echo json_encode(['error' => 'File not found']);
        exit;
    }

    // Get file size
    $fileSize = filesize($filePath);
    error_log("File size: " . $fileSize);

    // Handle download request
    if ($isDownload) {
        error_log("Processing download request");
        $originalName = preg_replace('/^[^_]*_/', '', $file); // Remove unique ID prefix
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $originalName . '"');
        header('Content-Length: ' . $fileSize);
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');
        readfile($filePath);
        exit;
    }

    // Regular streaming request
    // Set CORS headers to allow Chromecast to access the content
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $isChromecastLike = (
        strpos($userAgent, 'CrKey') !== false || 
        strpos($userAgent, 'Android') !== false || 
        strpos($userAgent, 'SmartTV') !== false ||
        strpos($userAgent, 'AppleTV') !== false ||
        strpos($userAgent, 'PlayStation') !== false ||
        strpos($userAgent, 'Xbox') !== false
    );
    
    if ($isChromecastLike) {
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
    }
    
    header('Content-Type: ' . ($type === 'audio' ? 'audio/mpeg' : 'video/mp4'));
    
    // Remove any previous output
    if (ob_get_level()) ob_end_clean();
    
    // Handle range requests for video streaming
    $range = isset($_SERVER['HTTP_RANGE']) ? $_SERVER['HTTP_RANGE'] : null;
    error_log("Range header: " . ($range ?? 'none'));
    
    // Set proper headers for video
    header('Accept-Ranges: bytes');
    header('Cache-Control: no-cache, must-revalidate');
    
    if ($range) {
        list($start, $end) = sscanf($range, 'bytes=%d-%d');
        if (!$end) $end = $fileSize - 1;
        
        header('HTTP/1.1 206 Partial Content');
        header("Content-Range: bytes $start-$end/$fileSize");
        header('Content-Length: ' . ($end - $start + 1));
        
        error_log("Streaming range: $start-$end/$fileSize");
        
        $fp = fopen($filePath, 'rb');
        fseek($fp, $start);
        $buffer = 1024 * 8;
        $remaining = $end - $start + 1;
        while ($remaining > 0 && !feof($fp)) {
            $readSize = min($buffer, $remaining);
            echo fread($fp, $readSize);
            $remaining -= $readSize;
            flush();
        }
        fclose($fp);
    } else {
        header('Content-Length: ' . $fileSize);
        error_log("Streaming full file");
        readfile($filePath);
    }
    exit;
}

function handleListFiles() {
    $type = $_GET['type'] ?? 'video';
    $dir = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos');
    
    $files = [];
    if (is_dir($dir)) {
        foreach (scandir($dir) as $file) {
            if ($file !== '.' && $file !== '..') {
                $path = $dir . '/' . $file;
                $files[] = [
                    'name' => $file,
                    'path' => $path,
                    'size' => filesize($path),
                    'modified' => date('Y-m-d H:i:s', filemtime($path))
                ];
            }
        }
    }
    
    header('Content-Type: application/json');
    echo json_encode(['files' => $files]);
}

function handleDeleteFile() {
    $data = json_decode(file_get_contents('php://input'), true);
    $fileName = $data['file'] ?? '';
    $type = $data['type'] ?? 'video';
    
    if (empty($fileName)) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing file parameter']);
        return;
    }

    $filePath = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos') . '/' . $fileName;
    
    $success = true;
    if (file_exists($filePath)) {
        $success = unlink($filePath);
    }
    
    if ($success) {
        echo json_encode(['success' => true]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to delete file']);
    }
}

function handleRenameFile() {
    $data = json_decode(file_get_contents('php://input'), true);
    $oldName = $data['oldName'] ?? '';
    $newName = $data['newName'] ?? '';
    $type = $data['type'] ?? 'video';
    
    if (empty($oldName) || empty($newName)) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing file names']);
        return;
    }
    
    $dir = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos');
    $oldPath = $dir . '/' . basename($oldName);
    $newPath = $dir . '/' . basename($newName);
    
    if (!file_exists($oldPath)) {
        http_response_code(404);
        echo json_encode(['error' => 'File not found']);
        return;
    }
    
    $success = rename($oldPath, $newPath);
    
    if ($success) {
        echo json_encode(['success' => true]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to rename file']);
    }
}

function handleGetStats() {
    header('Content-Type: application/json');
    echo json_encode(getStorageStats());
}

function getStorageStats() {
    $config = require 'config.php';
    $videoDir = $config['upload_path'] . 'videos';
    $audioDir = $config['upload_path'] . 'audios';
    
    $stats = [
        'totalSize' => 0,
        'videoSize' => 0,
        'audioSize' => 0,
        'fileCount' => 0,
        'videoCount' => 0,
        'audioCount' => 0,
        'storageLimit' => $config['storage_limit']
    ];
    
    // Calculate video stats
    if (is_dir($videoDir)) {
        foreach (scandir($videoDir) as $file) {
            if ($file !== '.' && $file !== '..') {
                $size = filesize($videoDir . '/' . $file);
                $stats['videoSize'] += $size;
                $stats['totalSize'] += $size;
                $stats['fileCount']++;
                $stats['videoCount']++;
            }
        }
    }
    
    // Calculate audio stats
    if (is_dir($audioDir)) {
        foreach (scandir($audioDir) as $file) {
            if ($file !== '.' && $file !== '..') {
                $size = filesize($audioDir . '/' . $file);
                $stats['audioSize'] += $size;
                $stats['totalSize'] += $size;
                $stats['fileCount']++;
                $stats['audioCount']++;
            }
        }
    }
    
    return $stats;
}

function handleBackupUpload() {
    $config = require 'config.php';
    $backupDir = $config['upload_path'] . 'backups/';
    
    if (!file_exists($backupDir)) {
        mkdir($backupDir, 0755, true);
    }

    $file = $_FILES['backup'];
    $format = $_POST['format'] ?? 'unknown';
    
    // Generate backup filename with timestamp
    $timestamp = date('Y-m-d_H-i-s');
    $newFileName = "backup_{$timestamp}.{$format}";
    $targetPath = $backupDir . $newFileName;
    
    if (move_uploaded_file($file['tmp_name'], $targetPath)) {
        // Clean up old backups after successful upload
        cleanupOldBackups();
        echo json_encode(['success' => true, 'fileName' => $newFileName]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to store backup']);
    }
}

function handleListBackups() {
    $config = require 'config.php';
    $backupDir = $config['upload_path'] . 'backups/';
    
    // Make sure the backups directory exists
    if (!file_exists($backupDir)) {
        mkdir($backupDir, 0755, true);
    }
    
    $backups = [];
    
    if (is_dir($backupDir)) {
        foreach (scandir($backupDir) as $file) {
            if ($file !== '.' && $file !== '..') {
                $path = $backupDir . '/' . $file;
                $backups[] = [
                    'name' => $file,
                    'size' => filesize($path),
                    'created' => date('Y-m-d H:i:s', filemtime($path)),
                    'format' => pathinfo($file, PATHINFO_EXTENSION)
                ];
            }
        }
    }
    
    // Sort by creation date, newest first
    usort($backups, function($a, $b) {
        return strtotime($b['created']) - strtotime($a['created']);
    });
    
    header('Content-Type: application/json');
    echo json_encode(['backups' => $backups]);
}

function handleGetBackup() {
    error_log("handleGetBackup called");
    $config = require 'config.php';
    $fileName = $_GET['file'] ?? '';
    
    error_log("Requested file: " . $fileName);
    
    if (empty($fileName)) {
        error_log("No filename provided");
        http_response_code(400);
        echo json_encode(['error' => 'Missing file parameter']);
        return;
    }
    
    $backupDir = $config['upload_path'] . 'backups/';
    error_log("Backup directory: " . $backupDir);
    
    if (!file_exists($backupDir)) {
        error_log("Creating backup directory");
        mkdir($backupDir, 0755, true);
    }
    
    $filePath = $backupDir . basename($fileName);
    error_log("Looking for backup file at: " . $filePath);
    
    if (!file_exists($filePath)) {
        error_log("Backup file not found: " . $filePath);
        http_response_code(404);
        echo json_encode(['error' => 'Backup not found']);
        return;
    }
    
    // Set appropriate headers based on file type
    $extension = pathinfo($fileName, PATHINFO_EXTENSION);
    $contentType = match($extension) {
        'json' => 'application/json',
        'sql' => 'application/sql',
        'zip' => 'application/zip',
        'csv' => 'text/csv',
        default => 'application/octet-stream'
    };
    
    header('Content-Type: ' . $contentType);
    header('Content-Disposition: attachment; filename="' . $fileName . '"');
    header('Content-Length: ' . filesize($filePath));
    
    // Remove any previous output
    if (ob_get_level()) ob_end_clean();
    
    readfile($filePath);
    exit;
}

function cleanupOldBackups() {
    $config = require 'config.php';
    $backupDir = $config['upload_path'] . 'backups/';
    $retentionDays = $config['backup_retention_days'];
    
    if (!is_dir($backupDir)) return;
    
    $files = scandir($backupDir);
    $now = time();
    
    foreach ($files as $file) {
        if ($file === '.' || $file === '..') continue;
        
        $filePath = $backupDir . '/' . $file;
        $fileAge = $now - filemtime($filePath);
        
        // Delete files older than retention period
        if ($fileAge > ($retentionDays * 86400)) {
            unlink($filePath);
            error_log("Deleted old backup: " . $file);
        }
    }
}

// Main execution
try {
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        exit(0);
    }

    error_log("Processing " . $_SERVER['REQUEST_METHOD'] . " request");

    // Check auth for all requests except OPTIONS
    if (!$auth->verifyToken()) {
        header('Content-Type: application/json');
        error_log("Authentication failed");
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        exit;
    }

    switch ($_SERVER['REQUEST_METHOD']) {
        case 'POST':
            if (isset($_FILES['file'])) {
                handleUpload();
            } elseif (strpos($_SERVER['REQUEST_URI'], '/delete') !== false) {
                handleDeleteFile();
            } elseif (strpos($_SERVER['REQUEST_URI'], '/rename') !== false) {
                handleRenameFile();
            } elseif (strpos($_SERVER['REQUEST_URI'], '/backup') !== false) {
                handleBackupUpload();
            }
            break;
            
        case 'GET':
            error_log("GET request received");
            error_log("URI contains 'backups': " . (strpos($_SERVER['REQUEST_URI'], '/backups') !== false ? 'yes' : 'no'));
            error_log("Has file parameter: " . (isset($_GET['file']) ? 'yes' : 'no'));
            
            if (strpos($_SERVER['REQUEST_URI'], '/backups') !== false) {
                if (isset($_GET['file'])) {
                    handleGetBackup();
                } else {
                    handleListBackups();
                }
            } elseif (isset($_GET['file'])) {
                handleFileRequest();
            } elseif (isset($_GET['type'])) {
                if ($_GET['type'] === 'stats') {
                    handleGetStats();
                } elseif (strpos($_SERVER['REQUEST_URI'], '/list') !== false) {
                    handleListFiles();  // This handles /api/storage/list?type=video
                } else {
                    handleListFiles();  // This handles /api/storage?type=video
                }
            } else {
                header('Content-Type: application/json');
                echo json_encode([
                    'status' => 'ok',
                    'message' => 'Video storage API endpoint'
                ]);
            }
            break;
            
        default:
            header('Content-Type: application/json');
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            exit;
    }
    
} catch (Exception $e) {
    header('Content-Type: application/json');
    error_log("Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

// At the very top of the file
error_log("Request received: " . $_SERVER['REQUEST_URI']);
error_log("Request method: " . $_SERVER['REQUEST_METHOD']);
error_log("Raw post data: " . file_get_contents('php://input'));
error_log("Files: " . print_r($_FILES, true));

// Add this to your existing mime type handling in handleFileRequest:
$mimeTypes = [
    'mp4' => 'video/mp4',
    'webm' => 'video/webm',
    'ogg' => 'video/ogg',
    'mp3' => 'audio/mpeg',
    'wav' => 'audio/wav',
    'm4a' => 'audio/mp4'
];

// At the top of the file, add more logging
error_log("Full Request URI: " . $_SERVER['REQUEST_URI']);
error_log("Query String: " . $_SERVER['QUERY_STRING']);
error_log("GET params: " . print_r($_GET, true));
?> 