<?php
class Auth {
    private $allowedDomains = [
        'localhost:3000',
        'helgehaukeland.com',
        'helgehaukeland.no',
        'www.helgehaukeland.com',
        'www.helgehaukeland.no'
    ];

    // Add method to detect Chromecast requests
    private function isChromecastRequest() {
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        return (
            strpos($userAgent, 'CrKey') !== false || // Chromecast
            strpos($userAgent, 'Android') !== false && strpos($userAgent, 'TV') !== false || // Android TV
            strpos($userAgent, 'SmartTV') !== false || // Smart TVs
            strpos($userAgent, 'AppleTV') !== false || // Apple TV
            strpos($userAgent, 'PlayStation') !== false || // PlayStation 
            strpos($userAgent, 'Xbox') !== false // Xbox
        );
    }

    private function generateNonce() {
        return bin2hex(random_bytes(16));
    }

    private function isXHR() {
        $headers = getallheaders();
        return isset($headers['X-Requested-With']) && 
               strtolower($headers['X-Requested-With']) === 'xmlhttprequest';
    }

    private function isMediaRequest() {
        return strpos($_SERVER['REQUEST_URI'], '.mp4') !== false ||
               strpos($_SERVER['REQUEST_URI'], '.webm') !== false ||
               strpos($_SERVER['REQUEST_URI'], '.m4v') !== false;
    }

    public function verifyToken() {
        // Only allow OPTIONS without auth
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            return true;
        }

        $headers = getallheaders();
        error_log("Headers: " . print_r($headers, true));
        error_log("GET params: " . print_r($_GET, true));
        error_log("Request Method: " . $_SERVER['REQUEST_METHOD']);
        
        // Check if request is coming from allowed domains
        $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        // Block requests from common download tools
        if (preg_match('/(wget|curl|python|ruby|java|wget|aria2|IDM|FDM)/i', $userAgent)) {
            error_log("Blocked download tool: " . $userAgent);
            return false;
        }

        $isAllowedOrigin = false;
        foreach ($this->allowedDomains as $domain) {
            if (strpos($origin, $domain) !== false || strpos($referer, $domain) !== false) {
                $isAllowedOrigin = true;
                break;
            }
        }

        // For media requests, ensure it's not a direct browser access
        if ($this->isMediaRequest()) {
            $isRange = isset($_SERVER['HTTP_RANGE']);
            $isHLSRequest = isset($headers['Accept']) && 
                           strpos($headers['Accept'], 'application/vnd.apple.mpegurl') !== false;
            $isDownloadRequest = isset($_GET['download']) && $_GET['download'] === 'true';
            $isChromecast = $this->isChromecastRequest();
            
            // Allow if it's a download request, range request, HLS request, or Chromecast
            if (!$isRange && !$isHLSRequest && !$isDownloadRequest && !$this->isXHR() && !$isChromecast) {
                error_log("Direct media access blocked - not a valid player request");
                http_response_code(403);
                echo json_encode(['error' => 'Direct access not allowed']);
                exit;
            }
        }

        // Try to get token from various sources
        $token = null;

        // Check Authorization header (case-insensitive)
        foreach ($headers as $name => $value) {
            if (strtolower($name) === 'authorization') {
                $token = str_replace('Bearer ', '', $value);
                break;
            }
        }
        
        // If no token in header, check query parameter
        if (!$token && isset($_GET['auth_token'])) {
            $token = $_GET['auth_token'];
        }

        // Allow HEAD requests with origin check
        if ($_SERVER['REQUEST_METHOD'] === 'HEAD' && $isAllowedOrigin) {
            return true;
        }

        error_log("Found token: " . ($token ?? 'none'));
        $isValid = $this->validateToken($token);

        // If this is a video request, ensure we have a valid token and origin
        if ($this->isMediaRequest()) {
            // Special case for Chromecast - only check token validity, not origin
            if ($this->isChromecastRequest()) {
                error_log("Chromecast request detected with User-Agent: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'none'));
                if (!$isValid) {
                    error_log("Chromecast request denied - invalid token");
                    return false;
                }
            } else if (!$isValid || !$isAllowedOrigin) {
                error_log("Video request denied - invalid token or origin");
                return false;
            }

            // Set headers to prevent caching and downloading
            header('Cache-Control: private, no-store, no-cache, must-revalidate, max-age=0');
            header('Pragma: no-cache');
            header('Content-Disposition: inline');
            header('Content-Transfer-Encoding: binary');
            header('X-Content-Type-Options: nosniff');
            header('X-Frame-Options: SAMEORIGIN');
            
            // Generate and set a nonce for additional security
            $nonce = $this->generateNonce();
            header('X-Content-Nonce: ' . $nonce);
        }

        return $isValid;
    }
    
    public function validateToken($token) {
        try {
            if (empty($token)) {
                error_log("Token is empty");
                return false;
            }
            // Add more robust token validation here
            if (strlen($token) < 32) {
                error_log("Token too short");
                return false;
            }
            error_log("Token validation successful");
            return true;
        } catch (Exception $e) {
            error_log("Token validation error: " . $e->getMessage());
            return false;
        }
    }
}
?> 