# Helge Haukeland CDN - Chromecast Support

This repository contains the secure media CDN for Helge Haukeland, with added support for Chromecast streaming.

## Chromecast Support Implementation

Chromecast support has been added to the secure media player, allowing videos to be cast to Chromecast devices while maintaining authentication security.

### How It Works

1. **Backend Authentication Enhancement**:
   - The `auth.php` file has been modified to detect Chromecast requests
   - Special handling has been added to validate tokens for Chromecast devices
   - CORS headers are set to allow Chromecast access when a valid token is provided

2. **JavaScript Helper**:
   - A `chromecast-helper.js` file has been created to facilitate Chromecast communication
   - This helper ensures that authentication tokens are passed in the URL when casting videos
   - The helper automatically stores and retrieves authentication tokens from local storage

### Implementation Steps

To add Chromecast support to your existing video player:

1. **Include the Chromecast Helper**:
   ```html
   <script src="/js/chromecast-helper.js"></script>
   ```

2. **Initialize the Helper**:
   ```javascript
   // Auto-initialized on DOMContentLoaded, or manually:
   const chromecastHelper = new ChromecastHelper({
     debug: true,  // Set to false in production
     authTokenName: 'auth_token'  // The name of your auth token parameter
   });
   ```

3. **Use the Helper with Your Video Player**:
   ```javascript
   // When loading a video, make sure the ChromecastHelper knows about it
   const videoElement = document.querySelector('video');
   chromecastHelper.updateVideoUrl(videoElement);
   
   // Or create a MediaInfo object for custom casting
   const mediaInfo = chromecastHelper.createMediaInfo(
     videoUrl,
     'video/mp4',
     { title: 'My Video' }
   );
   ```

4. **Sample Implementation**:
   A sample implementation is available in `js/sample-player.html` which demonstrates:
   - Loading videos with authentication
   - Storing and retrieving authentication tokens
   - Casting videos to Chromecast devices

### Server-side Changes

The server-side code has been updated to:

1. Detect Chromecast device requests based on User-Agent strings
2. Allow video requests from Chromecast devices with valid auth tokens
3. Include proper CORS headers to allow cross-origin requests from streaming devices
4. Extract auth tokens from URL parameters when Authorization headers aren't available

### Testing

To test the Chromecast functionality:

1. Load the sample player page
2. Enter a valid authentication token
3. Load a video
4. Click the Cast button to cast to a nearby Chromecast device
5. Verify the video plays on your TV

## Notes

- Ensure you have a valid authentication token
- The token will automatically be appended to the URL when casting
- The default Google Cast receiver application is used for playback
- Make sure your video is in a format supported by Chromecast (MP4 with H.264 video and AAC audio is recommended) 