<?php
// Error logging setup
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', '/home/natasun/helgehaukeland.com/secure_storage/logs/php-error.log');

// Debug logging
error_log("=== New Request ===");
error_log("Method: " . $_SERVER['REQUEST_METHOD']);
error_log("Origin: " . ($_SERVER['HTTP_ORIGIN'] ?? 'none'));
error_log("Auth: " . ($_SERVER['HTTP_AUTHORIZATION'] ?? 'none'));
error_log("Request URI: " . $_SERVER['REQUEST_URI']);

// Remove the global content type header
// header('Content-Type: application/json');

require_once 'auth.php';
$auth = new Auth();

function formatSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $size = $bytes;
    $unitIndex = 0;
    
    while ($size >= 1024 && $unitIndex < count($units) - 1) {
        $size /= 1024;
        $unitIndex++;
    }
    
    return round($size, 1) . ' ' . $units[$unitIndex];
}

function validateFile($file, $type) {
    $config = require 'config.php';
    
    // Check file size
    if ($file['size'] > $config['max_file_size']) {
        return [
            'valid' => false,
            'error' => 'File size exceeds limit of ' . formatSize($config['max_file_size'])
        ];
    }
    
    // Check file type
    $mimeType = mime_content_type($file['tmp_name']);
    if (!in_array($mimeType, $config['allowed_types'])) {
        return [
            'valid' => false,
            'error' => 'Invalid file type. Allowed types: ' . implode(', ', $config['allowed_types'])
        ];
    }
    
    // Check storage limit
    $stats = getStorageStats();
    if ($stats['totalSize'] + $file['size'] > $config['storage_limit']) {
        return [
            'valid' => false,
            'error' => 'Storage limit exceeded'
        ];
    }
    
    return ['valid' => true];
}

function handleUpload() {
    header('Content-Type: application/json');
    error_log("Starting handleUpload");
    
    if (!isset($_FILES['file'])) {
        error_log("No file uploaded");
        http_response_code(400);
        echo json_encode(['error' => 'No file uploaded']);
        return;
    }

    $type = $_POST['type'] ?? 'video';
    $file = $_FILES['file'];
    
    // Validate file
    $validation = validateFile($file, $type);
    if (!$validation['valid']) {
        http_response_code(400);
        echo json_encode(['error' => $validation['error']]);
        return;
    }
    
    // Continue with upload...
    $uploadDir = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos') . '/';
    
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $newFileName = uniqid() . '_' . basename($file['name']);
    $targetPath = $uploadDir . $newFileName;
    
    if (move_uploaded_file($file['tmp_name'], $targetPath)) {
        error_log("File uploaded successfully");
        echo json_encode(['fileName' => $newFileName]);
    } else {
        error_log("Upload failed: " . print_r(error_get_last(), true));
        http_response_code(500);
        echo json_encode(['error' => 'Failed to upload file']);
    }
}

function handleFileRequest() {
    error_log("Starting handleFileRequest");
    
    $file = $_GET['file'] ?? '';
    $type = $_GET['type'] ?? 'video';
    
    error_log("Requested file: " . $file);
    error_log("File type: " . $type);
    
    if (empty($file)) {
        header('Content-Type: application/json');
        http_response_code(400);
        echo json_encode(['error' => 'Missing file parameter']);
        exit;
    }

    // Clean filename and prevent directory traversal
    $file = basename($file);
    $filePath = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos') . '/' . $file;
    
    error_log("Looking for file at: " . $filePath);
    
    if (!file_exists($filePath)) {
        header('Content-Type: application/json');
        error_log("File not found: " . $filePath);
        http_response_code(404);
        echo json_encode(['error' => 'File not found']);
        exit;
    }

    // Set content type based on file type
    header('Content-Type: ' . ($type === 'audio' ? 'audio/mpeg' : 'video/mp4'));
    
    // Remove any previous output
    if (ob_get_level()) ob_end_clean();
    
    // Get file size
    $fileSize = filesize($filePath);
    error_log("File size: " . $fileSize);
    
    // Handle range requests for video streaming
    $range = isset($_SERVER['HTTP_RANGE']) ? $_SERVER['HTTP_RANGE'] : null;
    error_log("Range header: " . ($range ?? 'none'));
    
    // Set proper headers for video
    header('Content-Type: video/mp4');
    header('Accept-Ranges: bytes');
    header('Cache-Control: no-cache, must-revalidate');
    
    if ($range) {
        list($start, $end) = sscanf($range, 'bytes=%d-%d');
        if (!$end) $end = $fileSize - 1;
        
        header('HTTP/1.1 206 Partial Content');
        header("Content-Range: bytes $start-$end/$fileSize");
        header('Content-Length: ' . ($end - $start + 1));
        
        error_log("Streaming range: $start-$end/$fileSize");
        
        $fp = fopen($filePath, 'rb');
        fseek($fp, $start);
        $buffer = 1024 * 8;
        $remaining = $end - $start + 1;
        while ($remaining > 0 && !feof($fp)) {
            $readSize = min($buffer, $remaining);
            echo fread($fp, $readSize);
            $remaining -= $readSize;
            flush();
        }
        fclose($fp);
    } else {
        header('Content-Length: ' . $fileSize);
        error_log("Streaming full file");
        readfile($filePath);
    }
    exit;
}

function handleListFiles() {
    $type = $_GET['type'] ?? 'video';
    $dir = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos');
    
    $files = [];
    if (is_dir($dir)) {
        foreach (scandir($dir) as $file) {
            if ($file !== '.' && $file !== '..') {
                $path = $dir . '/' . $file;
                $files[] = [
                    'name' => $file,
                    'path' => $path,
                    'size' => filesize($path),
                    'modified' => date('Y-m-d H:i:s', filemtime($path))
                ];
            }
        }
    }
    
    header('Content-Type: application/json');
    echo json_encode(['files' => $files]);
}

function handleDeleteFile() {
    $data = json_decode(file_get_contents('php://input'), true);
    $fileName = $data['file'] ?? '';
    $type = $data['type'] ?? 'video';
    
    if (empty($fileName)) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing file parameter']);
        return;
    }

    $filePath = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos') . '/' . $fileName;
    
    $success = true;
    if (file_exists($filePath)) {
        $success = unlink($filePath);
    }
    
    if ($success) {
        echo json_encode(['success' => true]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to delete file']);
    }
}

function handleRenameFile() {
    $data = json_decode(file_get_contents('php://input'), true);
    $oldName = $data['oldName'] ?? '';
    $newName = $data['newName'] ?? '';
    $type = $data['type'] ?? 'video';
    
    if (empty($oldName) || empty($newName)) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing file names']);
        return;
    }
    
    $dir = '/home/natasun/helgehaukeland.com/secure_storage/' . 
        ($type === 'audio' ? 'audios' : 'videos');
    $oldPath = $dir . '/' . basename($oldName);
    $newPath = $dir . '/' . basename($newName);
    
    if (!file_exists($oldPath)) {
        http_response_code(404);
        echo json_encode(['error' => 'File not found']);
        return;
    }
    
    $success = rename($oldPath, $newPath);
    
    if ($success) {
        echo json_encode(['success' => true]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to rename file']);
    }
}

function handleGetStats() {
    header('Content-Type: application/json');
    echo json_encode(getStorageStats());
}

function getStorageStats() {
    $config = require 'config.php';
    $videoDir = $config['upload_path'] . 'videos';
    $audioDir = $config['upload_path'] . 'audios';
    
    $stats = [
        'totalSize' => 0,
        'videoSize' => 0,
        'audioSize' => 0,
        'fileCount' => 0,
        'videoCount' => 0,
        'audioCount' => 0,
        'storageLimit' => $config['storage_limit']
    ];
    
    // Calculate video stats
    if (is_dir($videoDir)) {
        foreach (scandir($videoDir) as $file) {
            if ($file !== '.' && $file !== '..') {
                $size = filesize($videoDir . '/' . $file);
                $stats['videoSize'] += $size;
                $stats['totalSize'] += $size;
                $stats['fileCount']++;
                $stats['videoCount']++;
            }
        }
    }
    
    // Calculate audio stats
    if (is_dir($audioDir)) {
        foreach (scandir($audioDir) as $file) {
            if ($file !== '.' && $file !== '..') {
                $size = filesize($audioDir . '/' . $file);
                $stats['audioSize'] += $size;
                $stats['totalSize'] += $size;
                $stats['fileCount']++;
                $stats['audioCount']++;
            }
        }
    }
    
    return $stats;
}

// Main execution
try {
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        exit(0);
    }

    error_log("Processing " . $_SERVER['REQUEST_METHOD'] . " request");

    // Check auth for all requests except OPTIONS
    if (!$auth->verifyToken()) {
        header('Content-Type: application/json');
        error_log("Authentication failed");
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        exit;
    }

    switch ($_SERVER['REQUEST_METHOD']) {
        case 'POST':
            if (isset($_FILES['file'])) {
                handleUpload();
            } elseif (strpos($_SERVER['REQUEST_URI'], '/delete') !== false) {
                handleDeleteFile();
            } elseif (strpos($_SERVER['REQUEST_URI'], '/rename') !== false) {
                handleRenameFile();
            }
            break;
            
        case 'GET':
            if (isset($_GET['file'])) {
                handleFileRequest();
            } elseif (isset($_GET['type']) && $_GET['type'] === 'stats') {
                handleGetStats();
            } elseif (isset($_GET['type'])) {
                handleListFiles();
            } else {
                header('Content-Type: application/json');
                echo json_encode([
                    'status' => 'ok',
                    'message' => 'Video storage API endpoint'
                ]);
            }
            break;
            
        default:
            header('Content-Type: application/json');
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            exit;
    }
    
} catch (Exception $e) {
    header('Content-Type: application/json');
    error_log("Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

// At the very top of the file
error_log("Request received: " . $_SERVER['REQUEST_URI']);
error_log("Request method: " . $_SERVER['REQUEST_METHOD']);
error_log("Raw post data: " . file_get_contents('php://input'));
error_log("Files: " . print_r($_FILES, true));

// Add this to your existing mime type handling in handleFileRequest:
$mimeTypes = [
    'mp4' => 'video/mp4',
    'webm' => 'video/webm',
    'ogg' => 'video/ogg',
    'mp3' => 'audio/mpeg',
    'wav' => 'audio/wav',
    'm4a' => 'audio/mp4'
];
?> 